<?php

if (!defined('ABSPATH')) {
    exit;
}

final class GP_Repository
{
    /** @var array<string,mixed> */
    private array $cfg;

    /**
     * @param array<string,mixed> $config
     */
    public function __construct(array $config = [])
    {
        $defaults = [
            'repo_url'             => GP_Get_Plugin::REPO_URL,
            'manifest_path'        => GP_Get_Plugin::MANIFEST_PATH,
            'package_path'         => GP_Get_Plugin::PACKAGE_PATH,
            'allowed_domains_path' => GP_Get_Plugin::ALLOWED_DOMAINS_PATH,
            'manifest_cache_key'   => GP_Get_Plugin::CACHE_KEY,
            'manifest_cache_ttl'   => GP_Get_Plugin::CACHE_TTL,
            'allowed_cache_key'    => GP_Get_Plugin::ALLOWED_CACHE_KEY,
            'allowed_cache_ttl'    => GP_Get_Plugin::ALLOWED_CACHE_TTL,
        ];

        $this->cfg = array_merge($defaults, $config);
    }

    public function invalidate_caches(): void
    {
        delete_transient((string) $this->cfg['manifest_cache_key']);
        delete_transient((string) $this->cfg['allowed_cache_key']);
    }

    public function repo_url(): string
    {
        $url = apply_filters('gp_repo_url', (string) $this->cfg['repo_url']);
        if (!is_string($url) || trim($url) === '') {
            $url = (string) $this->cfg['repo_url'];
        }

        return trailingslashit(esc_url_raw(trim((string) $url)));
    }

    public function manifest_url(): string
    {
        $path = apply_filters('gp_manifest_path', (string) $this->cfg['manifest_path']);
        if (!is_string($path) || trim($path) === '') {
            $path = (string) $this->cfg['manifest_path'];
        }

        return $this->repo_url() . ltrim((string) $path, '/');
    }

    public function package_url(string $slug, string $ver): string
    {
        $base = $this->repo_url() . ltrim((string) $this->cfg['package_path'], '/');
        return (string) add_query_arg(['slug' => $slug, 'ver' => $ver], $base);
    }

    /**
     * @return string[]
     */
    public function allowed_domains_url_candidates(): array
    {
        $base = $this->repo_url();

        $path = apply_filters('gp_allowed_domains_path', (string) $this->cfg['allowed_domains_path']);
        if (!is_string($path) || trim($path) === '') {
            $path = (string) $this->cfg['allowed_domains_path'];
        }

        $path = ltrim((string) $path, '/');

        return [
            $base . $path,
            $base . $path . '.json',
        ];
    }

    /**
     * @return array<string,mixed>
     */
    public function get_manifest_data(bool $force_refresh = false, int $timeout = 15): array
    {
        $cache_key = (string) $this->cfg['manifest_cache_key'];
        $cache_ttl = max(1, (int) $this->cfg['manifest_cache_ttl']);

        if (!$force_refresh) {
            $cached = get_transient($cache_key);
            if (is_array($cached) && (isset($cached['plugins']) || isset($cached['error']))) {
                return $cached;
            }
        }

        $url = $this->manifest_url();

        $res = wp_remote_get($url, [
            'timeout'     => max(5, (int) $timeout),
            'redirection' => 5,
            'sslverify'   => true,
            'headers'     => ['Accept' => 'application/json, text/plain;q=0.9, */*;q=0.8'],
            'user-agent'  => 'WordPress/' . get_bloginfo('version') . '; ' . home_url('/'),
        ]);

        if (is_wp_error($res)) {
            $out = ['error' => $res->get_error_message()];
            set_transient($cache_key, $out, 30);
            return $out;
        }

        $code = (int) wp_remote_retrieve_response_code($res);
        $body = (string) wp_remote_retrieve_body($res);

        if ($code < 200 || $code >= 300) {
            $out = ['error' => sprintf('HTTP %d from %s', $code, $url)];
            set_transient($cache_key, $out, 30);
            return $out;
        }

        $json = json_decode($body, true);
        if (!is_array($json)) {
            $out = ['error' => 'Manifest response is not valid JSON.'];
            set_transient($cache_key, $out, 30);
            return $out;
        }

        if (!isset($json['plugins']) || !is_array($json['plugins'])) {
            $out = ['error' => 'Manifest JSON has no "plugins" array.'];
            set_transient($cache_key, $out, 30);
            return $out;
        }

        set_transient($cache_key, $json, $cache_ttl);
        return $json;
    }

    /**
     * @return array<string,mixed>
     */
    public function get_allowed_domains_data(bool $force_refresh = false): array
    {
        $cache_key = (string) $this->cfg['allowed_cache_key'];
        $cache_ttl = max(1, (int) $this->cfg['allowed_cache_ttl']);

        if (!$force_refresh) {
            $cached = get_transient($cache_key);
            if (is_array($cached) && isset($cached['ok'], $cached['domains'])) {
                return $cached;
            }
        }

        $candidates = $this->allowed_domains_url_candidates();
        $last_error = '';

        foreach ($candidates as $url) {
            $res = wp_remote_get($url, [
                'timeout'     => 12,
                'redirection' => 3,
                'sslverify'   => true,
                'headers'     => ['Accept' => 'application/json, text/plain;q=0.9, */*;q=0.8'],
                'user-agent'  => 'WordPress/' . get_bloginfo('version') . '; ' . home_url('/'),
            ]);

            if (is_wp_error($res)) {
                $last_error = $res->get_error_message();
                continue;
            }

            $code = (int) wp_remote_retrieve_response_code($res);
            $body = (string) wp_remote_retrieve_body($res);

            if ($code < 200 || $code >= 300) {
                $last_error = sprintf('HTTP %d from %s', $code, $url);
                continue;
            }

            $json = json_decode($body, true);
            if (!is_array($json)) {
                $last_error = 'allowed_domains is not valid JSON.';
                continue;
            }

            $domains = [];
            if (isset($json['domains']) && is_array($json['domains'])) {
                foreach ($json['domains'] as $d) {
                    if (!is_string($d)) {
                        continue;
                    }
                    $n = GP_Utils::normalize_url_for_compare($d);
                    if ($n !== '') {
                        $domains[] = $n;
                    }
                }
            }

            $domains = array_values(array_unique($domains));

            $out = [
                'ok'      => true,
                'domains' => $domains,
                'source'  => $url,
                'error'   => '',
                'ts'      => time(),
            ];

            set_transient($cache_key, $out, $cache_ttl);
            return $out;
        }

        $out = [
            'ok'      => false,
            'domains' => [],
            'source'  => '',
            'error'   => $last_error !== '' ? $last_error : 'Failed to fetch allowed domains.',
            'ts'      => time(),
        ];

        set_transient($cache_key, $out, 60);
        return $out;
    }
}
